# Zalo Message Logging System

## 📝 Tổng quan

Hệ thống logging tự động ghi lại mọi hoạt động gửi tin nhắn Zalo vào file `log.txt`.

## 📂 File Structure

```
modules/zalo/
├── service.php         # Core functions (bao gồm zalo_log())
├── log.txt            # Log file (auto-generated)
├── view_log.php       # Web interface xem log
├── .gitignore         # Ignore log files khỏi Git
└── LOG_README.md      # File này
```

## 🔧 Function `zalo_log()`

### Cú pháp:
```php
zalo_log(string $message, string $level = 'INFO'): bool
```

### Parameters:
- `$message`: Nội dung log
- `$level`: Mức độ log (INFO, SUCCESS, ERROR, DEBUG)

### Ví dụ sử dụng:
```php
// Info log
zalo_log("User đang thực hiện thao tác X", 'INFO');

// Success log
zalo_log("Gửi tin nhắn thành công cho user ABC", 'SUCCESS');

// Error log
zalo_log("Lỗi kết nối API: Timeout", 'ERROR');

// Debug log
zalo_log("Payload: " . json_encode($data), 'DEBUG');
```

## 📊 Log Levels

| Level | Màu | Mục đích | Auto log PHP error |
|-------|-----|----------|-------------------|
| **INFO** | 🔵 Blue | Thông tin chung | ❌ Không |
| **SUCCESS** | 🟢 Green | Thao tác thành công | ❌ Không |
| **ERROR** | 🔴 Red | Lỗi, exception | ✅ Có |
| **DEBUG** | 🟣 Purple | Debug, chi tiết kỹ thuật | ❌ Không |

## 📋 Format Log

```
[YYYY-MM-DD HH:MM:SS] [LEVEL] Message
```

**Ví dụ thực tế:**
```
[2025-10-09 14:30:25] [INFO] Incoming Document Update - Attempting to send test message to user 70c8847eaf3d46631f2c
[2025-10-09 14:30:26] [DEBUG] Direct Send - Endpoint: https://bot-api.zapps.me/bot.../sendMessage
[2025-10-09 14:30:26] [DEBUG] Direct Send - Payload: {"chat_id":"70c8847eaf3d46631f2c","text":"Test Message"}
[2025-10-09 14:30:27] [SUCCESS] Direct Send - SUCCESS! Chat ID: 70c8847eaf3d46631f2c, Message sent
[2025-10-09 14:30:27] [SUCCESS] Message sent successfully to user 70c8847eaf3d46631f2c using bot 'Bot Test' (ID: 1)
```

## 🖥️ Xem Log

### Option 1: Web Interface (KHUYẾN NGHỊ)
```
http://localhost/Luna/modules/zalo/view_log.php
```

**Features:**
- ✅ Giao diện đẹp, dễ đọc
- ✅ Filter theo level (ALL, INFO, SUCCESS, ERROR, DEBUG)
- ✅ Hiển thị số dòng tùy chọn (50/100/200/500/1000)
- ✅ Statistics (tổng số log, success, error...)
- ✅ Color coding cho mỗi level
- ✅ Auto refresh 10s
- ✅ Responsive mobile-friendly

### Option 2: Đọc file trực tiếp
```php
$logs = file_get_contents(__DIR__ . '/modules/zalo/log.txt');
echo $logs;
```

### Option 3: Terminal/Command Line
```bash
# Xem toàn bộ
cat modules/zalo/log.txt

# Xem 50 dòng cuối
tail -n 50 modules/zalo/log.txt

# Xem realtime (live)
tail -f modules/zalo/log.txt

# Lọc chỉ ERROR
grep "ERROR" modules/zalo/log.txt

# Lọc theo ngày
grep "2025-10-09" modules/zalo/log.txt
```

## 🔄 Auto Logging

Các function sau **TỰ ĐỘNG** ghi log:

### 1. `zalo_set_last_error()`
Mỗi khi set error → Auto log ERROR level
```php
zalo_set_last_error("User not found");
// Auto ghi: [ERROR] User not found
```

### 2. `sendzalo()`
- Gửi thành công → Auto log SUCCESS
- Gửi thất bại → Auto log ERROR (qua zalo_set_last_error)

### 3. `sendzalo_direct()`
- Log DEBUG: Endpoint, Payload
- Log DEBUG: Response
- Log SUCCESS: Khi gửi thành công
- Log ERROR: Khi gửi thất bại

## 📁 File Management

### Rotation (Tùy chọn)
Nếu file log quá lớn, bạn có thể rotate:

```php
// Tự động rotate khi file > 5MB
$logFile = __DIR__ . '/log.txt';
if (filesize($logFile) > 5 * 1024 * 1024) {
    rename($logFile, __DIR__ . '/log_' . date('Y-m-d_His') . '.txt');
    file_put_contents($logFile, "# New log file\n");
}
```

### Clear Log
```php
// Xóa toàn bộ log
file_put_contents(__DIR__ . '/modules/zalo/log.txt', "# Log cleared\n");
```

### Backup
```bash
# Backup log
cp modules/zalo/log.txt modules/zalo/log_backup_$(date +%Y%m%d).txt
```

## 🔒 Security

- ✅ File log **KHÔNG** commit vào Git (có .gitignore)
- ✅ ERROR level **VẪN** ghi vào PHP error log (backup)
- ⚠️ **CẢNH BÁO**: Log có thể chứa thông tin nhạy cảm (user ID, token endpoint)
- 📌 **Khuyến nghị**: Đặt file log.txt ngoài public folder trong production

## 📊 Statistics & Analytics

Xem thống kê qua web interface hoặc:

```bash
# Đếm số log theo level
grep -c "INFO" modules/zalo/log.txt
grep -c "SUCCESS" modules/zalo/log.txt
grep -c "ERROR" modules/zalo/log.txt

# Tìm user nhận nhiều tin nhất
grep "sent successfully to user" modules/zalo/log.txt | awk '{print $10}' | sort | uniq -c | sort -rn | head -10
```

## 🎯 Best Practices

1. **Dùng đúng level:**
   - INFO: Thông tin flow (bắt đầu, kết thúc)
   - SUCCESS: Thành công rõ ràng
   - ERROR: Lỗi cần xử lý
   - DEBUG: Chi tiết kỹ thuật (payload, response)

2. **Message rõ ràng:**
   ```php
   // ❌ Bad
   zalo_log("Error", 'ERROR');
   
   // ✅ Good
   zalo_log("Failed to send message to user ABC123: HTTP 400 - Bad Request", 'ERROR');
   ```

3. **Include context:**
   ```php
   // ✅ Good
   zalo_log("Incoming Document #{$docId} - Sending notification to user {$userId}", 'INFO');
   ```

4. **Không log sensitive data:**
   ```php
   // ❌ Bad
   zalo_log("Token: " . $fullToken, 'DEBUG');
   
   // ✅ Good
   zalo_log("Token: " . substr($token, 0, 10) . "...", 'DEBUG');
   ```

## 🆘 Troubleshooting

### Không ghi được log?
```bash
# Kiểm tra quyền
ls -la modules/zalo/log.txt

# Set quyền write
chmod 666 modules/zalo/log.txt
```

### File log quá lớn?
```bash
# Xem kích thước
du -h modules/zalo/log.txt

# Rotate/Archive cũ
mv modules/zalo/log.txt modules/zalo/log_old.txt
touch modules/zalo/log.txt
```

### Cần search trong log?
```bash
# Tìm theo user ID
grep "70c8847eaf3d46631f2c" modules/zalo/log.txt

# Tìm lỗi trong ngày hôm nay
grep "$(date +%Y-%m-%d)" modules/zalo/log.txt | grep ERROR
```

## 📞 Support

- Xem log: `http://localhost/Luna/modules/zalo/view_log.php`
- Test send: `http://localhost/Luna/test_zalo.php`
- Documentation: `modules/zalo/USAGE.txt`


