<?php
// Outgoing document service: handle create/delete and utility requests
if (session_status() === PHP_SESSION_NONE) {
	session_start();
}

// Set timezone to Vietnam
date_default_timezone_set('Asia/Ho_Chi_Minh');

if (!defined('LUNA_APP')) {
	define('LUNA_APP', true);
}

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/messages.php';
$messages = load_messages();

function outgoing_validate_license(): void
{
	$settings = db_connection_settings();
	$expected = (string)($settings['license'] ?? '');
	$license = (string)($_POST['license'] ?? '');
	if ($expected === '' || $license !== $expected) {
		global $respond; $respond(false, 'Unauthorized request.');
	}
}

function outgoing_create(array $data): bool
{
	$conn = db_connect();
	if (!$conn) {
		return false;
	}

	// Note: For outgoing documents we store receiver into Issuer column
	$sql = "INSERT INTO GoDocument(
				ID_Symbol, TitleVi, Summary, DocType, TypeDoc,
				Quantity, Issuer, EffectiveDate,
				ID_Company, FileUrl, FileStorage,
				ID_by, Notes, CreatedAt
			) VALUES( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

	$params = [
		(string)($data['id_symbol'] ?? ''),
		(string)($data['title_vi'] ?? ''),
		(string)($data['summary'] ?? ''),
		(string)($data['doc_type'] ?? ''),
		(string)($data['type_doc'] ?? ''),
		($data['quantity'] === '' ? null : (int)$data['quantity']),
		(string)($data['receiver'] ?? ''), // mapped to Issuer column
		($data['effective_date'] ? date('Y-m-d', strtotime($data['effective_date'])) : null),
		($data['id_company'] === '' ? null : (int)$data['id_company']),
		(string)($data['file_url'] ?? ''),
		(string)($data['file_storage'] ?? ''),
		(string)($data['id_by'] ?? ''),
		(string)($data['notes'] ?? ''),
		($data['created_at'] ? date('Y-m-d H:i:s', strtotime($data['created_at'])) : date('Y-m-d H:i:s')),
	];

	$stmt = sqlsrv_query($conn, $sql, $params);
	$ok = $stmt !== false;
	
	if (!$ok) {
		$errors = sqlsrv_errors(SQLSRV_ERR_ERRORS);
		error_log('Outgoing create error: ' . print_r($errors, true));
		error_log('SQL: ' . $sql);
		error_log('Params: ' . print_r($params, true));
	}
	
	if ($stmt) sqlsrv_free_stmt($stmt);
	db_disconnect($conn);
	return $ok;
}

function outgoing_update(array $data): bool
{
	$conn = db_connect();
	if (!$conn) {
		return false;
	}

	$id = (int)($data['id'] ?? 0);
	if ($id <= 0) {
		db_disconnect($conn);
		return false;
	}

	// Note: For outgoing documents we store receiver into Issuer column
	$sql = "UPDATE GoDocument SET
				ID_Symbol = ?, TitleVi = ?, Summary = ?, DocType = ?, TypeDoc = ?,
				Quantity = ?, Issuer = ?, EffectiveDate = ?,
				ID_Company = ?, FileUrl = ?, FileStorage = ?,
				Notes = ?
			WHERE Id = ?";

	$params = [
		(string)($data['id_symbol'] ?? ''),
		(string)($data['title_vi'] ?? ''),
		(string)($data['summary'] ?? ''),
		(string)($data['doc_type'] ?? ''),
		(string)($data['type_doc'] ?? ''),
		($data['quantity'] === '' ? null : (int)$data['quantity']),
		(string)($data['receiver'] ?? ''), // mapped to Issuer column
		($data['effective_date'] ? date('Y-m-d', strtotime($data['effective_date'])) : null),
		($data['id_company'] === '' ? null : (int)$data['id_company']),
		(string)($data['file_url'] ?? ''),
		(string)($data['file_storage'] ?? ''),
		(string)($data['notes'] ?? ''),
		$id
	];

	$stmt = sqlsrv_query($conn, $sql, $params);
	$ok = $stmt !== false;
	
	if (!$ok) {
		$errors = sqlsrv_errors(SQLSRV_ERR_ERRORS);
		error_log('Outgoing update error: ' . print_r($errors, true));
		error_log('SQL: ' . $sql);
		error_log('Params: ' . print_r($params, true));
	}
	
	if ($stmt) sqlsrv_free_stmt($stmt);
	db_disconnect($conn);
	return $ok;
}

function outgoing_delete(int $id): bool
{
	$conn = db_connect();
	if (!$conn) {
		return false;
	}

	if ($id <= 0) {
		db_disconnect($conn);
		return false;
	}

	$sql = "DELETE FROM GoDocument WHERE Id = ?";
	$params = [$id];

	$stmt = sqlsrv_query($conn, $sql, $params);
	$ok = $stmt !== false;
	
	if (!$ok) {
		$errors = sqlsrv_errors(SQLSRV_ERR_ERRORS);
		error_log('Outgoing delete error: ' . print_r($errors, true));
		error_log('SQL: ' . $sql);
		error_log('Params: ' . print_r($params, true));
	}
	
	if ($stmt) sqlsrv_free_stmt($stmt);
	db_disconnect($conn);
	return $ok;
}

// Handle direct POST requests
if (isset($_SERVER['SCRIPT_FILENAME']) && @realpath(__FILE__) === @realpath($_SERVER['SCRIPT_FILENAME'])) {

	// Unified respond helper similar to incoming service
	$respond = function (bool $ok, string $message, array $data = []) use ($messages): void {
		$accept = strtolower((string)($_SERVER['HTTP_ACCEPT'] ?? ''));
		$isAjax = isset($_SERVER['HTTP_X_REQUESTED_WITH']) || strpos($accept, 'application/json') !== false;
		if ($isAjax) {
			header('Content-Type: application/json; charset=utf-8');
			$response = ['ok' => $ok, 'message' => $message];
			if (!empty($data)) { $response = array_merge($response, $data); }
			echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
		} else {
			if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? '')) === 'POST') {
				$param = $ok ? 'success' : 'error';
				$loc = '../index.php?outgoing&' . $param . '=' . rawurlencode($message);
				header('Location: ' . $loc);
			} else {
				http_response_code($ok ? 200 : 403);
				echo '<!doctype html><html lang="vi"><meta charset="utf-8"><title>Thông báo</title><body style="font-family:system-ui,Segoe UI,Arial">'
					. '<h2 style="color:#0d6efd;margin:24px 0">' . htmlspecialchars($message, ENT_QUOTES, 'UTF-8') . '</h2>'
					. '<p><a href="../index.php?outgoing">' . htmlspecialchars(($messages['control_nav_outgoing_documents'] ?? 'CÔNG VĂN ĐI'), ENT_QUOTES, 'UTF-8') . '</a></p>'
					. '</body></html>';
			}
		}
		exit;
	};

	if ($_SERVER['REQUEST_METHOD'] === 'POST') {
		$action = strtolower((string)($_POST['action'] ?? ''));
		
		// Handle get_file_url without license validation (read-only operation)
		if ($action === 'get_file_url') {
			header('Content-Type: application/json; charset=utf-8');
			
			$docId = (int)($_POST['id'] ?? 0);
			if ($docId <= 0) {
				echo json_encode(['success' => false, 'message' => 'ID tài liệu không hợp lệ']);
				exit;
			}
			
			try {
				$conn = db_connect();
				if (!$conn) {
					echo json_encode(['success' => false, 'message' => 'Không thể kết nối cơ sở dữ liệu']);
					exit;
				}
				
				$sql = "SELECT FileUrl FROM GoDocument WHERE Id = ?";
				$stmt = sqlsrv_query($conn, $sql, [$docId]);
				
				if ($stmt !== false) {
					if ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
						$fileUrl = (string)($row['FileUrl'] ?? '');
						if ($fileUrl) {
							echo json_encode(['success' => true, 'file_url' => $fileUrl]);
						} else {
							echo json_encode(['success' => false, 'message' => 'Tài liệu chưa có file PDF']);
						}
					} else {
						echo json_encode(['success' => false, 'message' => 'Không tìm thấy tài liệu']);
					}
					sqlsrv_free_stmt($stmt);
				} else {
					echo json_encode(['success' => false, 'message' => 'Lỗi truy vấn cơ sở dữ liệu']);
				}
				
				db_disconnect($conn);
			} catch (Throwable $e) {
				echo json_encode(['success' => false, 'message' => 'Lỗi hệ thống: ' . $e->getMessage()]);
			}
			exit;
		}
		
		// Validate license for write operations
		outgoing_validate_license();

        switch ($action) {
			case 'create':
				$title = trim((string)($_POST['title_vi'] ?? ''));
				$receiver = trim((string)($_POST['receiver'] ?? ''));
				$docType = trim((string)($_POST['doc_type'] ?? ''));
				$typeDoc = trim((string)($_POST['type_doc'] ?? ''));
				$effective = trim((string)($_POST['effective_date'] ?? ''));
				$quantity = trim((string)($_POST['quantity'] ?? ''));
				$company = trim((string)($_POST['id_company'] ?? ''));

				$errors = [];
                if ($title === '') { $errors[] = ($messages['incoming_require_title'] ?? 'Thiếu Tiêu đề.'); }
                if ($receiver === '') { $errors[] = ($messages['outgoing_require_receiver'] ?? 'Thiếu Cơ quan nhận.'); }
                if ($docType === '') { $errors[] = ($messages['incoming_require_type'] ?? 'Thiếu Thể loại.'); }
                if ($typeDoc === '') { $errors[] = ($messages['incoming_require_document_type'] ?? 'Thiếu Loại giấy tờ.'); }
                if ($effective === '') { $errors[] = ($messages['incoming_require_effective_date'] ?? 'Thiếu Ngày hiệu lực.'); }
                if ($quantity === '') { $errors[] = ($messages['incoming_require_quantity'] ?? 'Thiếu Số lượng.'); }
                if ($company === '') { $errors[] = ($messages['incoming_require_company'] ?? 'Thiếu Thuộc công ty.'); }

				if (!empty($errors)) {
                    // Number the lines similar to incoming service
                    $numbered = [];
                    foreach ($errors as $idx => $text) { $numbered[] = ($idx + 1) . '. ' . $text; }
                    $respond(false, implode("\n", $numbered));
				}

                $ok = outgoing_create([
					'id_symbol' => (string)($_POST['id_symbol'] ?? ''),
					'title_vi' => $title,
					'summary' => (string)($_POST['summary'] ?? ''),
					'doc_type' => (string)($_POST['doc_type'] ?? ''),
					'type_doc' => (string)($_POST['type_doc'] ?? ''),
					'quantity' => (string)($_POST['quantity'] ?? ''),
					'receiver' => $receiver,
					'effective_date' => $effective,
					'id_company' => (string)($_POST['id_company'] ?? ''),
					'id_category' => (string)($_POST['id_category'] ?? ''),
					'file_url' => (string)($_POST['file_url'] ?? ''),
					'file_storage' => (string)($_POST['file_storage'] ?? ''),
                    'id_by' => (string)($_POST['id_by'] ?? ($_SESSION['user']['code'] ?? '')),
                    'notes' => (string)($_POST['notes'] ?? ''),
                    'created_at' => (string)($_POST['created_at'] ?? date('Y-m-d H:i:s')),
				]);

                $msg = $ok ? ($messages['outgoing_create_success'] ?? 'Đã thêm công văn đi thành công.') : ($messages['outgoing_create_failed'] ?? 'Thêm mới thất bại.');
				$respond($ok, $msg);
				break;

			case 'update':
				$docId = (int)($_POST['id_document'] ?? 0);
				if ($docId <= 0) {
					$respond(false, 'ID tài liệu không hợp lệ');
				}

				$title = trim((string)($_POST['title_vi'] ?? ''));
				$receiver = trim((string)($_POST['receiver'] ?? ''));
				$docType = trim((string)($_POST['doc_type'] ?? ''));
				$typeDoc = trim((string)($_POST['type_doc'] ?? ''));
				$effective = trim((string)($_POST['effective_date'] ?? ''));
				$quantity = trim((string)($_POST['quantity'] ?? ''));
				$company = trim((string)($_POST['id_company'] ?? ''));

				$errors = [];
				if ($title === '') { $errors[] = ($messages['incoming_require_title'] ?? 'Thiếu Tiêu đề.'); }
				if ($receiver === '') { $errors[] = ($messages['outgoing_require_receiver'] ?? 'Thiếu Cơ quan nhận.'); }
				if ($docType === '') { $errors[] = ($messages['incoming_require_type'] ?? 'Thiếu Thể loại.'); }
				if ($typeDoc === '') { $errors[] = ($messages['incoming_require_document_type'] ?? 'Thiếu Loại giấy tờ.'); }
				if ($effective === '') { $errors[] = ($messages['incoming_require_effective_date'] ?? 'Thiếu Ngày hiệu lực.'); }
				if ($quantity === '') { $errors[] = ($messages['incoming_require_quantity'] ?? 'Thiếu Số lượng.'); }
				if ($company === '') { $errors[] = ($messages['incoming_require_company'] ?? 'Thiếu Thuộc công ty.'); }

				if (!empty($errors)) {
					$numbered = [];
					foreach ($errors as $idx => $text) { $numbered[] = ($idx + 1) . '. ' . $text; }
					$respond(false, implode("\n", $numbered));
				}

				$ok = outgoing_update([
					'id' => $docId,
					'id_symbol' => (string)($_POST['id_symbol'] ?? ''),
					'title_vi' => $title,
					'summary' => (string)($_POST['summary'] ?? ''),
					'doc_type' => (string)($_POST['doc_type'] ?? ''),
					'type_doc' => (string)($_POST['type_doc'] ?? ''),
					'quantity' => (string)($_POST['quantity'] ?? ''),
					'receiver' => $receiver,
					'effective_date' => $effective,
					'id_company' => (string)($_POST['id_company'] ?? ''),
					'file_url' => (string)($_POST['file_url'] ?? ''),
					'file_storage' => (string)($_POST['file_storage'] ?? ''),
					'notes' => (string)($_POST['notes'] ?? ''),
				]);

				$msg = $ok ? ($messages['outgoing_update_success'] ?? 'Đã cập nhật công văn đi thành công.') : ($messages['outgoing_update_failed'] ?? 'Cập nhật thất bại.');
				$respond($ok, $msg);
				break;

			case 'delete':
				$docId = (int)($_POST['id_document'] ?? 0);
				if ($docId <= 0) {
					$respond(false, 'ID tài liệu không hợp lệ');
				}

				$ok = outgoing_delete($docId);
				$msg = $ok ? ($messages['outgoing_delete_success'] ?? 'Đã xóa công văn đi thành công.') : ($messages['outgoing_delete_failed'] ?? 'Xóa thất bại.');
				$respond($ok, $msg);
				break;

			default:
				$respond(false, 'Invalid action');
		}
	} else {
		http_response_code(405);
		echo '<!doctype html><html lang="vi"><meta charset="utf-8"><title>Method Not Allowed</title><body style="font-family:system-ui,Segoe UI,Arial">'
			. '<h2 style="color:#b00020;margin:24px 0">' . htmlspecialchars($messages['error_invalid_access'] ?? 'Truy cập không hợp lệ.', ENT_QUOTES, 'UTF-8') . '</h2>'
			. '<p>' . htmlspecialchars($messages['error_use_interface'] ?? 'Vui lòng thao tác từ giao diện hệ thống.', ENT_QUOTES, 'UTF-8') . '</p>'
			. '<p><a href="../index.php?outgoing">' . htmlspecialchars($messages['control_nav_outgoing_documents'] ?? 'CÔNG VĂN ĐI', ENT_QUOTES, 'UTF-8') . '</a></p>'
			. '</body></html>';
	}
}

?>


