<?php
// Incoming document service: handle CRUD + direct requests
ob_start(); // Start output buffering to prevent any accidental output

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Set timezone to Vietnam
date_default_timezone_set('Asia/Ho_Chi_Minh');

if (!defined('LUNA_APP')) {
    define('LUNA_APP', true);
}

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/messages.php';
require_once __DIR__ . '/../modules/zalo/service.php';
$messages = load_messages();

// Error buffer (multi-code) like company
function incoming_set_last_error(string $code): void
{
    if (!isset($_SESSION['incoming_last_error']) || !is_array($_SESSION['incoming_last_error'])) {
        $_SESSION['incoming_last_error'] = [];
    }
    $_SESSION['incoming_last_error'][] = $code;
}

function incoming_get_last_error(): string
{
    $errs = $_SESSION['incoming_last_error'] ?? [];
    if (is_array($errs)) {
        $errs = array_values(array_unique(array_map('strval', $errs)));
        return implode(';', $errs);
    }
    return (string)$errs;
}

/**
 * Log incoming document activities to file
 * 
 * @param string $message Log message
 * @param string $level Log level: INFO, SUCCESS, ERROR, DEBUG
 * @return bool True if log was written successfully
 */
function incoming_log(string $message, string $level = 'INFO'): bool
{
    $logFile = __DIR__ . '/incoming/log.txt';
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[{$timestamp}] [{$level}] {$message}\n";
    
    // Ensure directory exists
    $dir = dirname($logFile);
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
    
    // Write to file
    $result = file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    
    // Also write to PHP error log for critical errors
    if ($level === 'ERROR') {
        error_log("Incoming: {$message}");
    }
    
    return $result !== false;
}

function incoming_validate_license(): void
{
    $settings = db_connection_settings();
    $expected = (string)($settings['license'] ?? '');
    $license = (string)($_POST['license'] ?? '');
    
    if ($expected === '' || $license !== $expected) {
        global $respond, $messages;
        $respond(false, $messages['error_unauthorized'] ?? 'Unauthorized request.');
    }
}

function incoming_create(array $data): bool
{
    $conn = db_connect();
    if (!$conn) {
        return false;
    }

    $sql = "INSERT INTO Document(
                ID_Symbol, TitleVi, Summary, DocType, TypeDoc,
                Quantity, Issuer, EffectiveDate, ExpiryDate,
                ID_Company, ID_Category, FileUrl, FileStorage, ID_by, Notes, CreatedAt
            ) VALUES( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

    $params = [
        (string)($data['id_symbol'] ?? ''),
        (string)($data['title_vi'] ?? ''),
        (string)($data['summary'] ?? ''),
        (string)($data['doc_type'] ?? ''),
        (string)($data['type_doc'] ?? ''),
        ($data['quantity'] === '' ? null : (int)$data['quantity']),
        (string)($data['issuer'] ?? ''),
        ($data['effective_date'] ?: null),
        ($data['expiry_date'] ?: null),
        ($data['id_company'] === '' ? null : (int)$data['id_company']),
        ($data['id_category'] === '' ? null : (int)$data['id_category']),
        (string)($data['file_url'] ?? ''),
        (string)($data['file_storage'] ?? ''),
        (string)($data['id_by'] ?? ''),
        (string)($data['notes'] ?? ''),
        ($data['created_at'] ?: date('Y-m-d H:i:s')),
    ];

    $stmt = sqlsrv_query($conn, $sql, $params);
    $ok = $stmt !== false;
    
    if (!$ok) {
        $errors = sqlsrv_errors();
        if ($errors) {
            error_log("SQL Insert Error: " . print_r($errors, true));
            error_log("SQL Query: " . $sql);
            error_log("SQL Params: " . print_r($params, true));
        }
    }
    
    if ($ok && isset($data['temp_file'])) {
        // Handle file rename if temp file exists
        $uploadDir = __DIR__ . '/../data/File/Document/';
        $tempFile = $uploadDir . $data['temp_file'];
        $finalFile = $uploadDir . ($data['document_id'] ?? 1) . '.pdf';
        
        if (file_exists($tempFile)) {
            rename($tempFile, $finalFile);
        }
    }
    
    if ($stmt) sqlsrv_free_stmt($stmt);
    db_disconnect($conn);
    return $ok;
}

function incoming_update(array $data): bool
{
    $conn = db_connect();
    if (!$conn) {
        return false;
    }

    $id = (int)($data['id'] ?? 0);
    if ($id <= 0) {
        db_disconnect($conn);
        return false;
    }

    $sql = "UPDATE Document SET
                ID_Symbol = ?, TitleVi = ?, Summary = ?, DocType = ?, TypeDoc = ?,
                Quantity = ?, Issuer = ?, EffectiveDate = ?, ExpiryDate = ?,
                ID_Company = ?, ID_Category = ?, FileUrl = ?, FileStorage = ?,
                Notes = ?
            WHERE Id = ?";

    $params = [
        (string)($data['id_symbol'] ?? ''),
        (string)($data['title_vi'] ?? ''),
        (string)($data['summary'] ?? ''),
        (string)($data['doc_type'] ?? ''),
        (string)($data['type_doc'] ?? ''),
        ($data['quantity'] === '' ? null : (int)$data['quantity']),
        (string)($data['issuer'] ?? ''),
        ($data['effective_date'] ?: null),
        ($data['expiry_date'] ?: null),
        ($data['id_company'] === '' ? null : (int)$data['id_company']),
        ($data['id_category'] === '' ? null : (int)$data['id_category']),
        (string)($data['file_url'] ?? ''),
        (string)($data['file_storage'] ?? ''),
        (string)($data['notes'] ?? ''),
        $id
    ];

    $stmt = sqlsrv_query($conn, $sql, $params);
    $ok = $stmt !== false;
    
    if (!$ok) {
        $errors = sqlsrv_errors();
        if ($errors) {
            error_log("SQL Update Error: " . print_r($errors, true));
            error_log("SQL Query: " . $sql);
            error_log("SQL Params: " . print_r($params, true));
        }
    }
    
    if ($stmt) sqlsrv_free_stmt($stmt);
    db_disconnect($conn);
    return $ok;
}

// Handle direct POST requests
if (isset($_SERVER['SCRIPT_FILENAME']) && @realpath(__FILE__) === @realpath($_SERVER['SCRIPT_FILENAME'])) {
    
    // Unified respond helper like company_service
    $respond = function (bool $ok, string $message, array $data = []) use ($messages): void {
        $accept = strtolower((string)($_SERVER['HTTP_ACCEPT'] ?? ''));
        $isAjax = isset($_SERVER['HTTP_X_REQUESTED_WITH']) || strpos($accept, 'application/json') !== false;
        
        if ($isAjax) {
            // Clear any buffered output before sending JSON
            while (ob_get_level() > 0) {
                ob_end_clean();
            }
            header('Content-Type: application/json; charset=utf-8');
            $response = ['ok' => $ok, 'message' => $message];
            if (!empty($data)) {
                $response = array_merge($response, $data);
            }
            echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit; // Stop execution after sending JSON
        } else {
            // For non-AJAX POSTs, redirect back with toast parameters (same as company)
            if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? '')) === 'POST') {
                $param = $ok ? 'success' : 'error';
                $loc = '../index.php?incoming&' . $param . '=' . rawurlencode($message);
                header('Location: ' . $loc);
            } else {
                http_response_code($ok ? 200 : 403);
                echo '<!doctype html><html lang="vi"><meta charset="utf-8"><title>Lỗi</title><body style="font-family:system-ui,Segoe UI,Arial">'
                . '<h2 style="color:#b00020;margin:24px 0">' . htmlspecialchars($message, ENT_QUOTES, 'UTF-8') . '</h2>'
                . '<p><a href="../index.php?incoming">' . htmlspecialchars(($messages['error_back_to_company'] ?? 'Quay lại trang Công văn đến'), ENT_QUOTES, 'UTF-8') . '</a></p>'
                . '</body></html>';
            }
        }
        exit;
    };
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $action = strtolower((string)($_POST['action'] ?? ''));
        
        // Skip license validation for get_file_url action
        if ($action !== 'get_file_url') {
            incoming_validate_license();
        }
        
        switch ($action) {
            case 'create':
                unset($_SESSION['incoming_last_error']);
                $title = trim((string)($_POST['title_vi'] ?? ''));
                $effective = trim((string)($_POST['effective_date'] ?? ''));
                $summary = trim((string)($_POST['summary'] ?? ''));
                $docType = trim((string)($_POST['doc_type'] ?? ''));
                $typeDoc = trim((string)($_POST['type_doc'] ?? ''));
                $quantity = trim((string)($_POST['quantity'] ?? ''));
                $issuer = trim((string)($_POST['issuer'] ?? ''));
                $idCompany = trim((string)($_POST['id_company'] ?? ''));
                $idCategory = trim((string)($_POST['id_category'] ?? ''));
                $idBy = trim((string)($_POST['id_by'] ?? ''));
                
                // Check ID_Symbol (số thứ tự) - first field in form
                $idSymbol = trim((string)($_POST['id_symbol'] ?? ''));
                
                $errors = [];
                // Only validate fields marked with asterisk (*) in the form
                if ($title === '') $errors[] = 'TITLE'; // Tiêu đề *
                if ($docType === '') $errors[] = 'DOCTYPE'; // Thể loại *
                if ($typeDoc === '') $errors[] = 'TYPEDOC'; // Loại giấy tờ *
                if ($quantity === '') $errors[] = 'QUANTITY'; // Số lượng *
                if ($issuer === '') $errors[] = 'ISSUER'; // Cơ quan cấp *
                if ($effective === '') $errors[] = 'EFFECTIVE'; // Ngày hiệu lực *
                if ($idCompany === '') $errors[] = 'COMPANY'; // Thuộc công ty *
                if ($idCategory === '') $errors[] = 'CATEGORY'; // Hạng mục *
                if ($idBy === '') $errors[] = 'IDBY'; // Mã số NV nhập *
                
                if (!empty($errors)) {
                    foreach ($errors as $error) {
                        incoming_set_last_error('VALIDATION:EMPTY:' . $error);
                    }
                    $detail = incoming_get_last_error();
                    $msg = $messages['incoming_require_data'] ?? 'Required data is missing';
                    $parts = [];
                    // Only show error messages for required fields (marked with asterisk)
                    if (strpos($detail, 'EMPTY:TITLE') !== false) { $parts[] = $messages['incoming_require_title'] ?? 'Title is required.'; }
                    if (strpos($detail, 'EMPTY:DOCTYPE') !== false) { $parts[] = $messages['incoming_require_type'] ?? 'Type is required.'; }
                    if (strpos($detail, 'EMPTY:TYPEDOC') !== false) { $parts[] = $messages['incoming_require_document_type'] ?? 'Document type is required.'; }
                    if (strpos($detail, 'EMPTY:QUANTITY') !== false) { $parts[] = $messages['incoming_require_quantity'] ?? 'Quantity is required.'; }
                    if (strpos($detail, 'EMPTY:ISSUER') !== false) { $parts[] = $messages['incoming_require_issuer'] ?? 'Issuing authority is required.'; }
                    if (strpos($detail, 'EMPTY:EFFECTIVE') !== false) { $parts[] = $messages['incoming_require_effective_date'] ?? 'Effective date is required.'; }
                    if (strpos($detail, 'EMPTY:COMPANY') !== false) { $parts[] = $messages['incoming_require_company'] ?? 'Company is required.'; }
                    if (strpos($detail, 'EMPTY:CATEGORY') !== false) { $parts[] = $messages['incoming_require_category'] ?? 'Category is required.'; }
                    if (strpos($detail, 'EMPTY:IDBY') !== false) { $parts[] = $messages['incoming_require_employee_id'] ?? 'Employee ID is required.'; }
                    
                    if ($parts) {
                        if (count($parts) >= 2) {
                            $numbered = [];
                            foreach ($parts as $idx => $text) { $numbered[] = ($idx + 1) . '. ' . $text; }
                            $msg = implode("\n", $numbered);
                        } else { 
                            $msg = implode("\n", $parts); 
                        }
                    } else { 
                        $msg = $messages['incoming_require_data'] ?? 'Required data is missing'; 
                    }
                    $respond(false, $msg);
                }
                $documentId = $_POST['document_id'] ?? 1;
                $ok = incoming_create([
                    'document_id' => $documentId,
                    'id_symbol' => $idSymbol,
                    'title_vi' => $title,
                    'summary' => $summary,
                    'doc_type' => $docType,
                    'type_doc' => $typeDoc,
                    'quantity' => $quantity,
                    'issuer' => $issuer,
                    'effective_date' => $effective,
                    'expiry_date' => $_POST['expiry_date'] ?? '',
                    'id_company' => $idCompany,
                    'id_category' => $idCategory,
                    'id_by' => $idBy,
                    'file_url' => $_POST['file_url'] ?? '',
                    'file_storage' => $_POST['file_storage'] ?? '',
                    'notes' => $_POST['notes'] ?? '',
                    'temp_file' => $_POST['temp_file'] ?? '',
                    'created_at' => $_POST['created_at'] ?? date('Y-m-d H:i:s'),
                ]);
                
                if ($ok) {
                    // Log successful import
                    $currentUser = (string)($_SESSION['user']['username'] ?? '');
                    incoming_log("Thêm tài liệu - ID: {$documentId}, Quantity: {$quantity}, User: {$currentUser}", 'IMPORT');
                }
                
                if (!$ok) {
                    incoming_set_last_error('DB:INSERT_FAILED');
                }
                $detail = incoming_get_last_error();
                unset($_SESSION['incoming_last_error']);
                $msg = $ok ? 'Đã thêm công văn đến.' : 'Thêm mới thất bại.';
                if (!$ok && $detail) {
                    if (strpos($detail, 'DB:INSERT_FAILED') !== false) {
                        $msg = 'Lỗi lưu dữ liệu.';
                    }
                }
                
                $respond($ok, $msg);
                break;
            case 'update':
                unset($_SESSION['incoming_last_error']);
                $docId = (int)($_POST['id_document'] ?? 0);
                if ($docId <= 0) {
                    $respond(false, $messages['incoming_invalid_id'] ?? 'ID tài liệu không hợp lệ');
                }

                $title = trim((string)($_POST['title_vi'] ?? ''));
                $effective = trim((string)($_POST['effective_date'] ?? ''));
                $docType = trim((string)($_POST['doc_type'] ?? ''));
                $typeDoc = trim((string)($_POST['type_doc'] ?? ''));
                $quantity = trim((string)($_POST['quantity'] ?? ''));
                $issuer = trim((string)($_POST['issuer'] ?? ''));
                $idCompany = trim((string)($_POST['id_company'] ?? ''));
                $idCategory = trim((string)($_POST['id_category'] ?? ''));
                
                $errors = [];
                if ($title === '') $errors[] = 'TITLE';
                if ($docType === '') $errors[] = 'DOCTYPE';
                if ($typeDoc === '') $errors[] = 'TYPEDOC';
                if ($quantity === '') $errors[] = 'QUANTITY';
                if ($issuer === '') $errors[] = 'ISSUER';
                if ($effective === '') $errors[] = 'EFFECTIVE';
                if ($idCompany === '') $errors[] = 'COMPANY';
                if ($idCategory === '') $errors[] = 'CATEGORY';
                
                if (!empty($errors)) {
                    foreach ($errors as $error) {
                        incoming_set_last_error('VALIDATION:EMPTY:' . $error);
                    }
                    $detail = incoming_get_last_error();
                    $parts = [];
                    if (strpos($detail, 'EMPTY:TITLE') !== false) { $parts[] = $messages['incoming_require_title'] ?? 'Title is required.'; }
                    if (strpos($detail, 'EMPTY:DOCTYPE') !== false) { $parts[] = $messages['incoming_require_type'] ?? 'Type is required.'; }
                    if (strpos($detail, 'EMPTY:TYPEDOC') !== false) { $parts[] = $messages['incoming_require_document_type'] ?? 'Document type is required.'; }
                    if (strpos($detail, 'EMPTY:QUANTITY') !== false) { $parts[] = $messages['incoming_require_quantity'] ?? 'Quantity is required.'; }
                    if (strpos($detail, 'EMPTY:ISSUER') !== false) { $parts[] = $messages['incoming_require_issuer'] ?? 'Issuing authority is required.'; }
                    if (strpos($detail, 'EMPTY:EFFECTIVE') !== false) { $parts[] = $messages['incoming_require_effective_date'] ?? 'Effective date is required.'; }
                    if (strpos($detail, 'EMPTY:COMPANY') !== false) { $parts[] = $messages['incoming_require_company'] ?? 'Company is required.'; }
                    if (strpos($detail, 'EMPTY:CATEGORY') !== false) { $parts[] = $messages['incoming_require_category'] ?? 'Category is required.'; }
                    
                    if ($parts) {
                        if (count($parts) >= 2) {
                            $numbered = [];
                            foreach ($parts as $idx => $text) { $numbered[] = ($idx + 1) . '. ' . $text; }
                            $msg = implode("\n", $numbered);
                        } else { 
                            $msg = implode("\n", $parts); 
                        }
                    } else { 
                        $msg = $messages['incoming_require_data'] ?? 'Required data is missing'; 
                    }
                    $respond(false, $msg);
                }

                $ok = incoming_update([
                    'id' => $docId,
                    'id_symbol' => (string)($_POST['id_symbol'] ?? ''),
                    'title_vi' => $title,
                    'summary' => (string)($_POST['summary'] ?? ''),
                    'doc_type' => $docType,
                    'type_doc' => $typeDoc,
                    'quantity' => $quantity,
                    'issuer' => $issuer,
                    'effective_date' => $effective,
                    'expiry_date' => $_POST['expiry_date'] ?? '',
                    'id_company' => $idCompany,
                    'id_category' => $idCategory,
                    'file_url' => $_POST['file_url'] ?? '',
                    'file_storage' => $_POST['file_storage'] ?? '',
                    'notes' => $_POST['notes'] ?? '',
                ]);

                if (!$ok) {
                    incoming_set_last_error('DB:UPDATE_FAILED');
                }
                $detail = incoming_get_last_error();
                unset($_SESSION['incoming_last_error']);
                $msg = $ok ? ($messages['incoming_update_success'] ?? 'Đã cập nhật công văn đến.') : ($messages['incoming_update_failed'] ?? 'Cập nhật thất bại.');
                if (!$ok && $detail) {
                    if (strpos($detail, 'DB:UPDATE_FAILED') !== false) {
                        $msg = $messages['incoming_update_db_error'] ?? 'Lỗi cập nhật dữ liệu.';
                    }
                }
                // Test Zalo send - Now works automatically!
              
                $respond($ok, $msg);
                break;
            case 'delete':
                $docId = (int)($_POST['id'] ?? 0);
                if ($docId <= 0) {
                    $respond(false, $messages['incoming_invalid_id'] ?? 'ID tài liệu không hợp lệ');
                    break;
                }

                try {
                    $conn = db_connect();
                    if (!$conn) {
                        $respond(false, $messages['incoming_error_db_connection'] ?? 'Không thể kết nối database');
                        break;
                    }

                    // Check if document exists
                    $checkSql = "SELECT Id FROM Document WHERE Id = ?";
                    $checkStmt = sqlsrv_query($conn, $checkSql, [$docId]);
                    if ($checkStmt === false) {
                        $respond(false, $messages['incoming_error_db_query'] ?? 'Lỗi truy vấn database');
                        break;
                    }

                    $exists = sqlsrv_fetch_array($checkStmt, SQLSRV_FETCH_ASSOC);
                    sqlsrv_free_stmt($checkStmt);

                    if (!$exists) {
                        $respond(false, $messages['incoming_error_not_found'] ?? 'Không tìm thấy tài liệu');
                        break;
                    }

                    // Delete document
                    $deleteSql = "DELETE FROM Document WHERE Id = ?";
                    $deleteStmt = sqlsrv_query($conn, $deleteSql, [$docId]);
                    if ($deleteStmt === false) {
                        $respond(false, $messages['incoming_error_delete_failed'] ?? 'Không thể xóa tài liệu');
                        break;
                    }

                    sqlsrv_free_stmt($deleteStmt);
                    db_disconnect($conn);

                    $respond(true, $messages['incoming_delete_success'] ?? 'Xóa tài liệu thành công');
                } catch (Throwable $e) {
                    $respond(false, ($messages['system_error'] ?? 'Lỗi hệ thống') . ': ' . $e->getMessage());
                }
                break;
            case 'get_file_url':
                $docId = (int)($_POST['id'] ?? 0);
                error_log("get_file_url called with docId: " . $docId);
                if ($docId <= 0) {
                    $respond(false, $messages['incoming_invalid_id'] ?? 'ID tài liệu không hợp lệ');
                    break;
                }
                
                try {
                    $conn = db_connect();
                    if (!$conn) {
                        $respond(false, $messages['incoming_error_db_connection'] ?? 'Không thể kết nối database');
                        break;
                    }
                    
                    $sql = "SELECT FileUrl FROM Document WHERE Id = ?";
                    $stmt = sqlsrv_query($conn, $sql, [$docId]);
                    if ($stmt === false) {
                        $respond(false, $messages['incoming_error_db_query'] ?? 'Lỗi truy vấn database');
                        break;
                    }
                    
                    $row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC);
                    sqlsrv_free_stmt($stmt);
                    db_disconnect($conn);
                    
                    if ($row && !empty($row['FileUrl'])) {
                        $respond(true, $messages['incoming_get_file_success'] ?? 'Thành công', ['file_url' => $row['FileUrl']]);
                    } else {
                        $respond(false, $messages['incoming_error_file_not_found'] ?? 'Không tìm thấy file');
                    }
                } catch (Throwable $e) {
                    $respond(false, ($messages['system_error'] ?? 'Lỗi hệ thống') . ': ' . $e->getMessage());
                }
                break;
            
            case 'export':
                $docId = (int)($_POST['document_id'] ?? 0);
                $quantity = (int)($_POST['export_quantity'] ?? 0);
                $recipient = trim((string)($_POST['export_recipient'] ?? ''));
                $reason = trim((string)($_POST['export_reason'] ?? ''));
                $exportedBy = (string)($_SESSION['user']['username'] ?? '');
                
                // Validation
                if ($docId <= 0) {
                    $respond(false, $messages['export_error_invalid_id'] ?? 'ID tài liệu không hợp lệ');
                    break;
                }
                if ($quantity <= 0) {
                    $respond(false, $messages['export_error_invalid_quantity'] ?? 'Số lượng xuất phải lớn hơn 0');
                    break;
                }
                if (empty($recipient)) {
                    $respond(false, $messages['export_error_no_recipient'] ?? 'Vui lòng chọn người nhận');
                    break;
                }
                if (empty($reason)) {
                    $respond(false, $messages['export_error_no_reason'] ?? 'Vui lòng nhập lý do xuất');
                    break;
                }
                
                try {
                    $conn = db_connect();
                    if (!$conn) {
                        $respond(false, $messages['export_error_db_connection'] ?? 'Không thể kết nối database');
                        break;
                    }
                    
                    // 1. Kiểm tra số lượng hiện có
                    $checkSql = "SELECT Quantity FROM Document WHERE Id = ?";
                    $checkStmt = sqlsrv_query($conn, $checkSql, [$docId]);
                    if ($checkStmt === false) {
                        db_disconnect($conn);
                        $respond(false, $messages['export_error_db_query'] ?? 'Lỗi truy vấn database');
                        break;
                    }
                    
                    $row = sqlsrv_fetch_array($checkStmt, SQLSRV_FETCH_ASSOC);
                    sqlsrv_free_stmt($checkStmt);
                    
                    if (!$row) {
                        db_disconnect($conn);
                        $respond(false, $messages['export_error_not_found'] ?? 'Không tìm thấy tài liệu');
                        break;
                    }
                    
                    $currentQuantity = (int)($row['Quantity'] ?? 0);
                    if ($quantity > $currentQuantity) {
                        db_disconnect($conn);
                        $errorMsg = $messages['export_error_exceed_quantity'] ?? 'Số lượng xuất vượt quá số lượng hiện có';
                        $respond(false, $errorMsg . ' (' . $currentQuantity . ')');
                        break;
                    }
                    
                    // 2. Giảm số lượng trong bảng Document và lưu người nhận
                    $updateSql = "UPDATE Document SET Quantity = Quantity - ?, ID_Export = ?, Notes = ? WHERE Id = ?";
                    $updateStmt = sqlsrv_query($conn, $updateSql, [$quantity, $recipient, $reason, $docId]);
                    if ($updateStmt === false) {
                        db_disconnect($conn);
                        $respond(false, $messages['export_error_update_failed'] ?? 'Không thể cập nhật số lượng tài liệu');
                        break;
                    }
                    sqlsrv_free_stmt($updateStmt);
                    
                    db_disconnect($conn);
                    
                    // Log successful export
                    incoming_log("Xuất tài liệu - ID: {$docId}, Quantity: {$quantity}, Recipient: {$recipient}, Reason: {$reason}, User: {$exportedBy}", 'EXPORT');
                    
                    // Replace placeholders in success message
                    $successMsg = $messages['export_success'] ?? 'Xuất file thành công. Đã xuất {quantity} file cho {recipient}';
                    $successMsg = str_replace('{quantity}', $quantity, $successMsg);
                    $successMsg = str_replace('{recipient}', $recipient, $successMsg);
                    $respond(true, $successMsg);
                } catch (Throwable $e) {
                    $respond(false, ($messages['system_error'] ?? 'Lỗi hệ thống') . ': ' . $e->getMessage());
                }
                break;
            
            case 'import':
                $docId = (int)($_POST['document_id'] ?? 0);
                $quantity = (int)($_POST['import_quantity'] ?? 0);
                $sender = trim((string)($_POST['import_sender'] ?? ''));
                $reason = trim((string)($_POST['import_reason'] ?? ''));
                $importedBy = (string)($_SESSION['user']['username'] ?? '');
                
                // Validation
                if ($docId <= 0) {
                    $respond(false, $messages['import_error_invalid_id'] ?? 'ID tài liệu không hợp lệ');
                    break;
                }
                if ($quantity <= 0) {
                    $respond(false, $messages['import_error_invalid_quantity'] ?? 'Số lượng nhập phải lớn hơn 0');
                    break;
                }
                if (empty($sender)) {
                    $respond(false, $messages['import_error_no_sender'] ?? 'Vui lòng chọn người gửi');
                    break;
                }
                if (empty($reason)) {
                    $respond(false, $messages['import_error_no_reason'] ?? 'Vui lòng nhập lý do nhập');
                    break;
                }
                
                try {
                    $conn = db_connect();
                    if (!$conn) {
                        $respond(false, $messages['import_error_db_connection'] ?? 'Không thể kết nối database');
                        break;
                    }
                    
                    // 1. Kiểm tra tài liệu có tồn tại không
                    $checkSql = "SELECT Id FROM Document WHERE Id = ?";
                    $checkStmt = sqlsrv_query($conn, $checkSql, [$docId]);
                    if ($checkStmt === false) {
                        db_disconnect($conn);
                        $respond(false, $messages['import_error_db_query'] ?? 'Lỗi truy vấn database');
                        break;
                    }
                    
                    $exists = sqlsrv_fetch_array($checkStmt, SQLSRV_FETCH_ASSOC);
                    sqlsrv_free_stmt($checkStmt);
                    
                    if (!$exists) {
                        db_disconnect($conn);
                        $respond(false, $messages['import_error_not_found'] ?? 'Không tìm thấy tài liệu');
                        break;
                    }
                    
                    // 2. Tăng số lượng trong bảng Document
                    $updateSql = "UPDATE Document SET Quantity = Quantity + ? WHERE Id = ?";
                    $updateStmt = sqlsrv_query($conn, $updateSql, [$quantity, $docId]);
                    if ($updateStmt === false) {
                        db_disconnect($conn);
                        $respond(false, $messages['import_error_update_failed'] ?? 'Không thể cập nhật số lượng tài liệu');
                        break;
                    }
                    sqlsrv_free_stmt($updateStmt);
                    
                    db_disconnect($conn);
                    
                    // Log successful import
                    incoming_log("Nhập tài liệu - ID: {$docId}, Quantity: {$quantity}, Sender: {$sender}, Reason: {$reason}, User: {$importedBy}", 'IMPORT');
                    
                    // Replace placeholders in success message
                    $successMsg = $messages['import_success'] ?? 'Nhập file thành công. Đã nhập {quantity} file từ {sender}';
                    $successMsg = str_replace('{quantity}', $quantity, $successMsg);
                    $successMsg = str_replace('{sender}', $sender, $successMsg);
                    $respond(true, $successMsg);
                } catch (Throwable $e) {
                    $respond(false, ($messages['system_error'] ?? 'Lỗi hệ thống') . ': ' . $e->getMessage());
                }
                break;
            
            default:
                $respond(false, $messages['error_invalid_action'] ?? 'Invalid action');
        }
    } else {
        http_response_code(405);
        echo '<!doctype html><html lang="vi"><meta charset="utf-8"><title>Method Not Allowed</title><body style="font-family:system-ui,Segoe UI,Arial">'
            . '<h2 style="color:#b00020;margin:24px 0">' . htmlspecialchars($messages['error_invalid_access'] ?? 'Truy cập không hợp lệ.', ENT_QUOTES, 'UTF-8') . '</h2>'
            . '<p>' . htmlspecialchars($messages['error_use_interface'] ?? 'Vui lòng thao tác từ giao diện hệ thống.', ENT_QUOTES, 'UTF-8') . '</p>'
            . '<p><a href="../index.php?incoming">' . htmlspecialchars($messages['error_back_to_company'] ?? 'Quay lại trang Công ty', ENT_QUOTES, 'UTF-8') . '</a></p>'
            . '</body></html>';
    }
}



